{
Copyright  1999 by Delphi 5 Developer's Guide - Xavier Pacheco and Steve Teixeira
}

unit MemMap;

interface

uses Windows, SysUtils, Classes;

type
  EMMFError = class(Exception);

  TMemMapFile = class
  private
    FFileName: String;    // File name of the mapped file.
    FSize: Longint;       // Size of the mapped view
    FFileSize: Longint;   // Actual File Size
    FFileMode: Integer;   // File access mode
    FFileHandle: Integer; // File handle
    FMapHandle: Integer;  // Handle to the file mapping object.
    FData: PByte;         // Pointer to the file's data
    FMapNow: Boolean;     // Determines whether or
                          //   not to map view of immediately.
    procedure AllocFileHandle;
    { Retrieves a handle to the disk file. }
    procedure AllocFileMapping;
    { Retrieves a file-mapping object handle }
    procedure AllocFileView;
    { Maps a view to the file }
    function GetSize: Longint;
    { Returns the size of the mapped view }
  public
    constructor Create(FileName: String; FileMode: integer;
                       Size: integer; MapNow: Boolean); virtual;
    destructor Destroy; override;
    procedure FreeMapping;
    property Data: PByte read FData;
    property Size: Longint read GetSize;
    property FileName: String read FFileName;
    property FileHandle: Integer read FFileHandle;
    property MapHandle: Integer read FMapHandle;
  end;

implementation

constructor TMemMapFile.Create(FileName: String; FileMode: integer;
                               Size: integer; MapNow: Boolean);
{ Creates Memory Mapped view of FileName file.
  FileName: Full pathname of file.
  FileMode: Use fmXXX constants.
  Size: size of memory map.  Pass zero as the size to use the
        file's own size.
}
begin

  { Initialize private fields }
  FMapNow := MapNow;
  FFileName := FileName;
  FFileMode := FileMode;

  AllocFileHandle;  // Obtain a file handle of the disk file.
  { Assume file is < 2 gig  }

  FFileSize := GetFileSize(FFileHandle, Nil);
  FSize := Size;

  try
    AllocFileMapping; // Get the file mapping object handle.
  except
    on EMMFError do
    begin
      CloseHandle(FFileHandle);  // close file handle on error
      FFileHandle := 0;          // set handle back to 0 for clean up
      raise;                     // re-raise exception
    end;
  end;
  if FMapNow then
    AllocFileView;  // Map the view of the file
end;

destructor TMemMapFile.Destroy;
begin

  if FFileHandle <> 0 then
    CloseHandle(FFileHandle); // Release file handle.
    
  { Release file mapping object handle }
  if FMapHandle <> 0 then
    CloseHandle(FMapHandle);

  FreeMapping; { Unmap the file mapping view . }
  inherited Destroy;
end;

procedure TMemMapFile.FreeMapping;
{ This method unmaps the view of the file from this process's address
  space. }
begin
  if FData <> Nil then
  begin
    UnmapViewOfFile(FData);
    FData := Nil;
  end;
end;

function TMemMapFile.GetSize: Longint;
begin
  if FSize <> 0 then
    Result := FSize
  else
    Result := FFileSize;
end;

procedure TMemMapFile.AllocFileHandle;
{ creates or opens disk file before creating memory mapped file }
begin
  if FFileMode = fmCreate then
    FFileHandle := FileCreate(FFileName)
  else
    FFileHandle := FileOpen(FFileName, FFileMode);

  if FFileHandle < 0 then
    raise EMMFError.Create('Falhou a abrir o arquivo!');
end;

procedure TMemMapFile.AllocFileMapping;
var
  ProtAttr: DWORD;
begin
  if FFileMode = fmOpenRead then  // obtain correct protection attribute
    ProtAttr := Page_ReadOnly
  else
    ProtAttr := Page_ReadWrite;
  { attempt to create file mapping of disk file.
    Raise exception on error. }
  FMapHandle := CreateFileMapping(FFileHandle, Nil, ProtAttr,
      0, FSize, Nil);
  if FMapHandle = 0 then
    raise EMMFError.Create('Falhou quando tentou mapar o arquivo');
end;

procedure TMemMapFile.AllocFileView;
var
  Access: Longint;
begin
  if FFileMode = fmOpenRead then // obtain correct file mode
    Access := File_Map_Read
  else
    Access := File_Map_All_Access;
  FData := MapViewOfFile(FMapHandle, Access, 0, 0, FSize);
  if FData = Nil then
    raise EMMFError.Create('Falhou a mapear o arquivo!');
end;

end.
